[
//
// [ChartFX_Designer]
//
// Lasso Summit 2007
// Author: Fletcher Sandbeck
// Copyright 2007 OmniPilot Software
// 
//
// This data type allows a Chart FX template to be formatted within Lasso.
//
// A Chart FX template is an XML file which consists of many attributes. Each
// attribute is represented by a single tag within the file. Some attributes
// have simple values and others have compound values represented by sub-tags.
//
// Parameters
// 
// Some parameters are shared among several of the tags and have pre-defined
// values. These include: 
//
// -Name='Font Name' always specifies the name of a font, -Size specifies the
// size of the font in points, and -Bold or -Italic can be used to add that
// attribute to the font. Either -Color or -TextColor may be used to set the
// color of text depending on the tag.
//
// -Color can be specified as an HTML color value like #ff0000. An additional
// pair specifies the transparency of the color from #ff000000 for transparent
// to #ff0000ff for solid. Colors can also be specified by name including White,
// Light Gray, Gray, Dark Gray, Black, Red, Pink, Orange, Yellow, Green,
// Magenta, Cyan, Blue, and Transparent.
//
// -Alignment can have a value of Near, Center, Far, or Spread. Near usually
// specifies that the element should be aligned closer to the 0 point on the
// axis. Spread specifies that the element should be fit best into the layout.
// Different values will be the default depending on what is being aligned.
//
// Line styles are set with -Width to specify the width of the line in pixels,
// -StartCap and -EndCap can have a value of Butt (default), Round, or Square,
// and -Style can have a value of Solid (default), Dash, DashDot, or DashDotDot.
//
// Name/Value parameters such as -Gallery='pie' can be passed to most tags
// including the creator. The attribute 'GALLERY' will be created with the value
// 'pie'. This allows attributes which are not anticipated by this tag set to be
// used. Sub-attributes can be added by including a map as the value of an
// attribute. Sub-attribute values can be specified as 'font'=(Map:
// 'name'='Verdana', 'size'=9), however most sub-attributes are handled
// specially by the individual tags of the type.
//
// Constructor
//
// [ChartFX_Designer: -Gallery='pie'] sets the basic parameters of the chart.
// Requires a -Gallery parameter which specifies the type of chart. -Gallery
// defaults to Bar and can also be set to Lines, Area, Scatter, Pie, Curve,
// Pareto, Step, HiLowClose, Surface, Radar, Cube, Doughnut, Pyramid, Bubble,
// OpenHiLowClose, Candlestick, Contour, CurveArea, or Gantt. -Height and -Width
// can be set here or using member tags before the chart is rendered.  Other
// parameters which are usually set here include -Border to turn on a border
// around data objects with options -BorderColor and -BorderEffect with values
// None, Raised, Light, Dark, Opposite, Shadow. -Cluster specifies that values
// from different series should be stacked rather than shown side by side.
// -CylSides gives elements of a bar graph a rounded appearance if set to a high
// (>25) number. -InsideColor controls the background color of the graph itself.
// The default font for the chart can be set using -Name, -Size, and -Bold or
// -Italic. All labels and titles in the chart will use this font unless
// instructed otherwise. -Palette can be set to one of Windows, EarthTones,
// ModernBusiness, Alternate, HighContrast, Pastels, Sky, Adventure, Mesa, or
// Vivid.
//
// Rendering
//
// [ChartFX_Designer->SetLassoData] accepts an array of arrays containing the
// data to be rendered.  The [ChartFX_Records] tag returns data in the proper
// format for this tag based on the current inline.  See the description of that
// tag for more information.
//
// [ChartFX_Designer->Render] returns an HTML anchor tag referencing the
// rendered chart.  This tag works identically to the [ChartFX->Render] tag. 
//
// [ChartFX_Designer->Data] returns the raw data of the rendered chart image. 
// This tag works identically to the [ChartFX->Data] tag.
//
// [ChartFX_Designer->SetHeight] and [ChartFX_Designer->SetWidth] can be used if
// the height and width of the chart were not set in the creator.
// 
// Note - The template for the chart is set using the remainder of the tags of
// the type.  If a preformatted template is to be used then the [ChartFX] type
// should be used directly.
// 
// Series
//
// ChartFX will automatically use the chart preferences for each data series and
// will automatically determine the number of data series unless one of the tags
// here is used.  For example, if the data contains three fields the first will
// map to the X axis and the second and third will be seen as data series. 
// However, if only one series is defined using [ChartFX_Designer->Series] then
// only the first data series will be rendered.
//
// [ChartFX_Designer->(Series: 1)] sets general preferences for the specified
// data series.  -Color assigns a specific color to the data series.  -Gallery
// allows the data series to be drawn using a different chart type from other
// data series.  -Stacked controls whether the data series is stacked with other
// series or not. -Visible=false can be used to hide a data series.  -LineStyle
// and -LineWidth can be set.
//
// [ChartFX_Designer->(SeriesLabel: 1)] turns on the point labels for the
// specified data series.  Options include -Angle, -Color, and the font options
// -Name, -Size, -Bold, and -Italic.
//
// [ChartFX_Designer->(SeriesMarker: 1, 'shape')] sets the desired marker for
// the specified data series in line graphs.  The -Shape parameter can be one of
// None, Rect, Circle, Triangle, Diamond, Marble, HorizLine, VertLine, Cross,
// InvertedTriangle, X, or Many.  The -Color of the marker can be specified. 
// Also accepts -Size and -Step parameters.
//
// Chart Appearance
//
// [ChartFX_Designer->(AddTitle: 'text')] adds a text title to the chart.
// Multiple titles can be added. The tag requires one parameter which is the
// text of the title. The tag accepts font parameters -Name, -Size, etc. The
// location of the title can be set with -Alignment, -Right, -Center, -Left, or
// -Top for position. The color of the title can be set with -Color and the
// background color with -BackColor. If -DrawingArea is specified then the title
// will be the width of the chart, otherwise it will be the full with of the
// background.
//
// [ChartFX_Designer->(Set3D)] displays the chart in 3D.  -View3D can be set to
// False if you want to render the chart in 3D, but within a flat frame.  In
// addition, several parameter which are specific to the 3D display of charts
// can be set here. -View3DLight can be set to Fixed or Realistic.  -View3DDepth
// can be set to a percentage value to control the depth of 3D bar chart solids
// relative to their thickness.  By default, -View3DDepth is set to 100%, a
// value of 200 will make the bars twice as deep as they are thick. -Perspective
// can be set to a value from 0 (the default) to 100 to change the ratio of the
// size of the front to the back of the chart.  -AngleX and -AngleY can be set
// to rotate the chart.  -WallWidth controls the width of the wall which
// surrounds the chart.
//
// [ChartFX_Designer->(SetMarker: 'shape')] sets the desired marker for all
// series in line graphs (by default each series will use a different shape). 
// The -Shape parameter can be one of None, Rect, Circle, Triangle, Diamond,
// Marble, HorizLine, VertLine, Cross, InvertedTriangle, X, or Many.  Also
// accepts -Size and -Step parameters.
//
// [ChartFX_Designer->(SetBackground: 'color')] sets the background of the chart
// to a solid color.
//
// [ChartFX_Designer->(SetBackground: 'color', 'color')] sets the background of
// the chart to a gradient.  Also accepts parameter -Type with a value of
// Horizontal, Vertical, BackwardDiagonal, ForwardDiagonal, Radial, or Angled. 
// For the last option the angle can be specified with -Angle.
//
// [ChartFX_Designer->(SetBorder: 'type')] sets the border of the chart to the
// desired type.  Built-in types include Inner, Outer, Flat, SunkenInner,
// RaisedInner, SunkenOuter, RaisedOuter, DarkOuter, and LightOuter. The tag
// also supports a number of image-based types including Embed, Emboss, Arch,
// ArchThick, ArchStroke, ArchStrokeThick, Solid, SolidRivet, SolidLines,
// SolidLinesRivet, Open, OpenRivet, OpenLines, OpenLinesRivet, Colonial, Slide,
// Rounded, RoundedCaption, RoundedShadow, RoundedCaptionShadow, Aqua,
// Butterfly, Gel, Museum, and Pulsar. Some of these types accept an optional
// -Color parameter.  
//
// [ChartFX_Designer->(Legend)] shows the legend for the values in each series.
// Accepts -Name, -Size, -Bold, -Italic, and -TextColor parameters.  -Internal
// specifies that the legend should be compact and -External specifies the
// legend should extend to the sides of the chart.  The position of the legend
// is controlled by -Top, -Left, -Right, or -Bottom.  -Color specifies the
// background color for the legend. 
//
// [ChartFX_Designer->(SeriesLegend)] shows the legend for the data series.
// Accepts -Name, -Size, -Bold, -Italic, and -TextColor parameters.  -Internal
// specifies that the legend should be compact and -External specifies the
// legend should extend to the sides of the chart.  The position of the legend
// is controlled by -Top, -Left, -Right, or -Bottom.  -Color specifies the
// background color for the legend.
//
// Axis Control
//
// These tags set the appearance of the axes for the chart. Each tag requires
// the name of an axis as its first parameter. The axis will usually be "x" or
// "y", but may also be "x2" or "y2" for certain chart types. The -Step
// parameter value is shared for each axis so setting it in any tag will affect
// the value for all tags. The "minor" tags also share a -Step value.
//
// [ChartFX_Designer->(Axis: 'x')] sets basic parameters for an axis. Common
// parameters include -Max, -Min, -Inverted, -ScaleUnit, -Staggered, -Step,
// -ForceZero, and -LogBase. If -Visible=False is specified then the axis, its
// labels, and title are all hidden.
//
// [ChartFX_Designer->(AxisLabel: 'x')] sets options for the automatically
// generated labels of the axis. -Step controls how often a label is shown.
// -Angle controls the angle at which the labels are drawn.  The font of labels
// can be controlled with -Name, -Size, -Bold, and -Italic.  -Color controls the
// label color. -Position controls where labels are drawn relative to the axis.
// -Format can be set to None, Number, Currency, Scientific, Percentage, Date,
// LongDate, Time, or DateTime and -Decimals controls how many places past the
// decimal point will be shown. -First allows the first desired label to be
// specified.  -ScaleUnit allows the entire axis to be divided by the specified
// number and -LabelValue allows the entire axis to be multiplied by the
// specified number.
//
// [ChartFX_Designer->(AxisTitle: 'x', 'Title)] displays a title by an axis. 
// The font and color can be controlled with -Name, -Size, -Bold, -Italic, and
// -Color. The -Position of the title can also be specified.
//
// [ChartFX_Designer->(AxisLine: 'x')] sets preferences for the axis line
// itself. Options include -Color, -EndCap, -StartCap, -Style, -Width.
//
// [ChartFX_Designer->(AxisStripe: 'x', 'color')] allows automatic striping of
// the data. The background will alternate between the inner color set in the
// constructor and the -Color specified here. Also accepts the common -Step parameter.
//
// [ChartFX_Designer->(AxisGrid: 'x')] displays a grid from the axis across the
// chart.  Options include -Color, -EndCap, -StartCap, -Style, and -Width. Also
// accepts the common -Step parameter.
//
// [ChartFX_Designer->(AxisTick: 'x')] displays a tick on each step. The -Tick
// parameter accepts options None, Outside, Cross, and Inside. Also accepts the
// common -Step parameter.
//
// [ChartFX_Designer->(AxisMinorGrid: 'x')] displays a grid based on the minor
// step from the axis across the chart.  Options include -Color, -EndCap,
// -StartCap, -Style, and -Width. Also accepts the common minor -Step parameter.
//
// [ChartFX_Designer->(AxisMinorTick: 'x')] displays a tick on each minor step.
// The -Tick parameter accepts options None, Outside, Cross, and Inside. Also
// accepts the common minor -Step parameter.
//
// Data Highlights 
//
// [ChartFX_Designer->(AddConstantLine: 'x', '3', 'text')] adds a constant line
// to the chart. Multiple constant lines can be added. Each line is drawn across
// the chart from either the X or Y axis. The tag requires two parameters which
// is the axis for the line and the value at which the line should be drawn. The
// tag also accepts a -Title parameter, -TextColor, and font parameters. The
// location of the title can be set with -Alignment, -LineAlignment, and
// -OutsideText to draw text in the axis rather than on the chart. The line
// style can be set with -Color, -StartCap, -EndCap, -Width, and -Style.
//
// [ChartFX_Designer->(AddStripe: 'x', -From=3, -To=5)] adds a stripe to the
// graph extending from the specified axis.  The strip will go from the -From
// value to the -To value and can have an optional -Color.
//
// Utility Tags
//
// [ChartFX_Designer->(Set: -Name='value')] is a utility tag which accepts the
// same types of parameters as the creator, but either sets or overrides the
// attribute within the Chart FX template. Multiple attributes can be set with
// this tag. If an attribute is given a new value then the new value completely
// replaces the old value (sub-attributes will not be merged).
//

Define_Type('Designer', -Namespace='ChartFX');
		Local('attr' = Map);
		Local('addl' = Map);
		Local('cfx' = ChartFX);
		Define_Tag('getfont');
			Local: '_font' = Map;
			If: Local_Defined('name');
				#_font->Insert('name' = @#name);
				Params->RemoveAll('name') & RemoveAll(-name) & RemoveAll(#name);
			/If;
			If: Local_Defined('size');
				#_font->Insert('size' = @#size);
				Params->RemoveAll('size') & RemoveAll(-size) & RemoveAll(#size);
			/If;
			If: Local_Defined('bold');
				#_font->Insert('bold' = (if_null: Local('bold'), True));
				Params->RemoveAll('bold') & RemoveAll(-bold) & RemoveAll(#bold);
			/If;
			If: Local_Defined('italic');
				#_font->Insert('italic' = (if_null: Local('italic'), True));
				Params->RemoveAll('italic') & RemoveAll(-italic) & RemoveAll(#italic);
			/If;
			If (#_font->Size > 0);
				#attr->Insert('font'=@#_font);
			/If;
		/Define_Tag;
		Define_Tag('getparams');
			Iterate(Params, Local('temp_param'));
				If: #temp_param->IsA('pair');
					If: #temp_param->First->WasKeyword;
						#attr->Insert(#temp_param->First - '-' = #temp_param->Second);
					Else;
						#attr->Insert(#temp_param->First = #temp_param->Second);
					/If;
				Else: #temp_param->WasKeyword;
					#attr->Insert(#temp_param - '-' = True);
				Else: #temp_param->(IsA: 'string');
					#attr->Insert(#temp_param = True);
				/If;
			/Iterate;
		/Define_Tag;
		Define_Tag('getnamed');
			Iterate(#_names, Local('_name'));
				If: #_name->IsA('string') && Local_Defined(#_name);
					#attr->Insert(#_name = If_Null(Local(#_name),True));
					Params->RemoveAll(#_name) & RemoveAll('-' + #_name) & RemoveAll(Local(#_name));
				Else: #_name->IsA('pair') && Local_Defined(#_name->First);
					#attr->Insert(#_name->Second = If_Null(Local(#_name->First),True));
					Params->RemoveAll(#_name->First) & RemoveAll('-' + #_name->First) & RemoveAll(Local(#_name->First));
				/If;
			/Iterate;
		/Define_Tag;
		Define_Tag('tagname', -Required='name', -Required='open');
			If: #open && Self->'addl' >> #name;
				Local: 'output' = '<' + String_UpperCase(Encode_XML(#name));
				Iterate: Self->'addl'->Find(#name), Local('param');
					#output += ' ' + Encode_XML(#param->first) + '="' + Encode_XML(#param->second) + '"';
				/Iterate;
				#output += '>';
				Return: @#output;
			Else;
				Return: '<' + (#open ? '' | '/') + String_UpperCase(Encode_XML(#name)) + '>';
			/If;
		/Define_Tag;
		Define_Tag('format', -Required='attr', -Required='level');
			Local('output' = Array());
			Iterate(#attr->Keys()->Sort &, Local('temp_name'));
				Local('temp_value' = #attr->Find(#temp_name));
				Select(#temp_value->Type());
					Case('boolean');
						#output->Insert(('\t' * #level) + Self->TagName(#temp_name,True) + (#temp_value == True ? 'true' | 'false') + Self->TagName(#temp_name,False));
					Case('map');
						#temp_value->Size == 0 ? Loop_Continue;
						#output->Insert(('\t' * #level) + Self->TagName(#temp_name,True));
						#output->Merge(Self->Format(#temp_value, #level + 1));
						#output->Insert(('\t' * #level) + Self->TagName(#temp_name,False));
					Case('array');
						Local('temp_output') = Array;
						Iterate(#temp_value, Local('temp_item'));
							#temp_item->Size == 0 ? Loop_Continue;
							If: #temp_name == 'axis' ||  #temp_name == 'series';
								#temp_output->Insert(('\t' * (#level + 1)) + '<ITEM index="' + (Loop_Count - 1) + '">');
							Else;
								#temp_output->Insert(('\t' * (#level + 1)) + '<ITEM>');
							/If;
							#temp_output->Merge(Self->Format(#temp_item, #level + 2));
							#temp_output->Insert(('\t' * (#level + 1)) + '</ITEM>');
						/Iterate;
						If(#temp_output->Size > 0);
							#output->Insert(('\t' * #level) + Self->TagName(#temp_name,True));
							#output->Merge(#temp_output);
							#output->Insert(('\t' * #level) + Self->TagName(#temp_name,False));
						/If;
					Case;
						If: #temp_name == 'axis';
							#temp_value === 'y' ? #temp_value = 0;
							#temp_value === 'y2' ? #temp_value = 1;
							#temp_value === 'x' ? #temp_value = 2;
							#temp_value === 'x2' ? #temp_value = 3;
						Else: #temp_name == 'startcap' || #temp_name == 'endcap';
							#temp_value == 'butt' ? #temp_value = 0;
							#temp_value == 'round' ? #temp_value = 1;
							#temp_value == 'square' ? #temp_value = 2;
						Else: #temp_name == 'palette' && #temp_value !>> '.';
							#temp_value >> 'Window' ? #temp_value = 'Default.Windows';
							#temp_value >> 'Earth' ? #temp_value = 'Default.EarthTones';
							#temp_value >> 'Business' ? #temp_value = 'Default.ModernBusiness';
							#temp_value >> 'Alt' ? #temp_value = 'Default.Alternate';
							#temp_value >> 'Contrast' ? #temp_value = 'HighContrast.HighConstrast';
							#temp_value >> 'Pastel' ? #temp_value = 'DarkPastels.Pastels';
							#temp_value >> 'Sky' ? #temp_value = 'Nature.Sky';
							#temp_value >> 'Adventure' ? #temp_value = 'Nature.Adventure';
							#temp_value >> 'Mesa' ? #temp_value = 'Mesa.Mesa';
							#temp_value >> 'Vivid' ? #temp_value = 'Vivid.Vivid';
						/If;
						#output->Insert(('\t' * #level) + Self->TagName(#temp_name,True) + Encode_XML(#temp_value) + Self->TagName(#temp_name,False));
				/Select;
			/Iterate;
			Return(@#output);
		/Define_Tag;
	Define_Tag('onCreate');
		Local('attr' = (Map:
				'allowdrag'=false,
				'allowedit'=false,
				'bordereffect'='none',
				'chart3d'=false,
				'contextmenus'=false,
				'font'=(map: 'name'='Verdna', 'size'=9.0),
				'gallery'='bar',
				'htmltag'='Auto',
				'palette'='Default.ChartFX6',
				'pointlabels'=false,
				'showtips'=false,
				'height'=480,
				'width'=640,
				'axis'=(array: map, map, (map: 'labelangle'=90), map),
			));
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('onConvert');
		Return(Self->Template);			
	/Define_Tag;
	Define_Tag('Data');
		Fail_If(Self->'cfx'->Properties->Second->(Find: 'data') == Null, -9965, 'The ->Data tag is only supported in Lasso 8.5.2 or 8.1.1 or higher');
		Self->'cfx'->SetWidth(Integer(Self->'attr'->Find('width')));
		Self->'cfx'->SetHeight(Integer(Self->'attr'->Find('height')));
		Self->'cfx'->SetTemplateStr(Self->Template);
		Return(Self->'cfx'->Data);
	/Define_Tag;
	Define_Tag('Serve');
		Fail_If(Self->'cfx'->Properties->Second->(Find: 'data') == Null, -9965, 'The ->Serve tag is only supported in Lasso 8.5.2 or 8.1.1 or higher');
		Self->'cfx'->SetWidth(Integer(Self->'attr'->Find('width')));
		Self->'cfx'->SetHeight(Integer(Self->'attr'->Find('height')));
		Self->'cfx'->SetTemplateStr(Self->Template);
		Content_Type: 'image/png';
		Content_Body: Self->'cfx'->Data;
		Params !>> -NoAbort ? Abort;
	/Define_Tag;
	Define_Tag('Render');
		Self->'cfx'->SetWidth(Integer(Self->'attr'->Find('width')));
		Self->'cfx'->SetHeight(Integer(Self->'attr'->Find('height')));
		Self->'cfx'->SetTemplateStr(Self->Template);
		Return(Self->'cfx'->Render);
	/Define_Tag;
	Define_Tag('Template');
		Local('output' = Array());
		#output->Insert('<?xml version="1.0" encoding="UTF-8"?>');
		#output->Insert('<CFX6>');
		#output->Merge(Self->Format(Self->'attr', 1));
		#output->Insert('</CFX6>');
		Return(#output->Join('\r\n'));
	/Define_Tag;
	Define_Tag('_UnknownTag');
		Self->'cfx'->SetWidth(Integer(Self->'attr'->Find('width')));
		Self->'cfx'->SetHeight(Integer(Self->'attr'->Find('height')));
		If(Self->'cfx'->Properties->Second >> Tag_Name);
			Return(Self->'cfx'->Properties->Second->Find(Tag_Name)->Run(-Owner=Self->'cfx', -Params=Params, -Name=Tag_Name));
		/If;
	/Define_Tag;
	Define_Tag('SetWidth', -Required='width');
		Self->Set(-Width=Integer(#width));
	/Define_Tag;
	Define_Tag('SetHeight', -Required='height');
		Self->Set(-Height=Integer(#height));
	/Define_Tag;
	Define_Tag('Set');
		Local: 'attr' = Map;
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('SetFont', -optional='name', -optional='size');
		Local: 'attr' = Map;
		If: Local_Defined('name');
			#attr->Insert('name' = @#name);
			Params->RemoveAll('name') & RemoveAll(-name) & RemoveAll(#name);
		/If;
		If: Local_Defined('size');
			#attr->Insert('size' = @#size);
			Params->RemoveAll('size') & RemoveAll(-size) & RemoveAll(#size);
		/If;
		Iterate(Params, Local('temp_param'));
			If: #temp_param->IsA('pair');
				If: #temp_param->First->WasKeyword;
					#attr->Insert(#temp_param->First - '-' = #temp_param->Second);
				Else;
					#attr->Insert(#temp_param->First = #temp_param->Second);
				/If;
			Else: #temp_param->WasKeyword;
				#attr->Insert(#temp_param - '-' = True);
			Else: #temp_param->(IsA: 'string');
				#attr->Insert(#temp_param = True);
			/If;
		/Iterate;
		Self->'attr'->Insert('font' = @#attr);
	/Define_Tag;
	Define_Tag('SetBackground', -Optional='ColorFrom', -Optional='ColorTo', -Optional='Angle', -Optional='Type');
		If: Params->Size == 1 && Local_Defined('ColorFrom') || Local_Defined('Color');
			Self->'attr'->Remove('backobject');
			Local_Defined('colorfrom') ? Self->'attr'->Insert('backcolor'=Local(#colorfrom));
			Local_Defined('color') ? Self->'attr'->Insert('backcolor'=Local(#color));
			Return;
		/If;
		Local: 'attr' = Map;
		Local: '_names' = Array('colorfrom', 'colorto', 'type', 'angle');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		#attr >> 'colorfrom' ? #attr->Insert('color0' = #attr->Find('colorfrom')) & Remove('colorfrom');
		#attr >> 'colorto' ? #attr->Insert('color1' = #attr->Find('colorto')) & Remove('colorto');
		#attr >> 'color1' && #attr !>> 'color0' ? #attr->Insert('color0' = #attr->Find('color1'));
		#attr >> 'angle' && #attr !>> 'type' ? #attr->Insert('type' = 'angled');
		#attr->Insert('colorcount' = (integer: #attr >> 'color0') + (Integer: #attr >> 'color1'));
		Self->'addl'->Insert('backobject'=(Map: 'type'='SoftwareFX.ChartFX.Borders.GradientBackground', 'assembly'='ChartFX.Borders'));
		Self->'attr'->Insert('backobject' = @#attr);
	/Define_Tag;
	Define_Tag('SetBorder', -Required='type', -Optional='color');
		Local: 'attr' = Map;
		Local: '_names' = Array('type', 'color');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		If(#type >> 'Inner' || #type >> 'Outer' || #type >> 'Flat');
			Self->'addl'->Insert('borderobject'=(Map: 'type'='SoftwareFX.ChartFX.Borders.DefaultBorder', 'assembly'='ChartFX.Borders'));
		Else;
			Self->'addl'->Insert('borderobject'=(Map: 'type'='SoftwareFX.ChartFX.Borders.ImageBorder', 'assembly'='ChartFX.Borders'));
		/If;
		Self->'attr'->Insert('borderobject' = @#attr);
	/Define_Tag;
	Define_Tag('Set3D');
		Local('attr' = (Map:
				'chart3d' = true,
				'view3d' = true,
			));
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('SetMarker', -Optional='shape');
		Local('attr' = Map);
		Local('_names' = Array('shape'='markershape','size'='markersize','step'='markerstep'));
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('AddPointLabels');
		Local: 'attr' = Map;
		#attr->Insert('pointlabels'=True);
		Local: '_names' = Array('angle'='labelangle', 'color'='pointlabelcolor');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		#attr >> 'font' ? #attr->Insert('pointlabelfont' = #attr->Find('font')) & Remove('font');
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('AddTitle', -Required='text');
		Local: 'attr' = Map;
		Local: '_names' = Array('text', 'top', 'bottom', 'left', 'right');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		#attr >> 'top' ? #attr->Insert('area' = 'top') & Remove('top');
		#attr >> 'bottom' ? #attr->Insert('area' = 'bottom') & Remove('bottom');
		#attr >> 'left' ? #attr->Insert('area' = 'left') & Remove('left');
		#attr >> 'right' ? #attr->Insert('area' = 'right') & Remove('right');
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr' !>> 'titles' ? Self->'attr'->Insert('titles'=Array);
		Self->'attr'->Find('titles')->Insert(@#attr);
	/Define_Tag;
	Define_Tag('AddConstantLine', -Required='axis', -Required='value', -Optional='text');
		Local: 'attr' = Map;
		Local: '_names' = Array('axis', 'value', 'text');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr' !>> 'constantlines' ? Self->'attr'->Insert('constantlines'=Array);
		Self->'attr'->Find('constantlines')->Insert(@#attr);
	/Define_Tag;
	Define_Tag('AddStripe', -Required='axis', -Required='from', -required='to', -Optional='color');
		Local: 'attr' = Map;
		Local: '_names' = Array('axis', 'from', 'to', 'color');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr' !>> 'stripes' ? Self->'attr'->Insert('stripes'=Array);
		Self->'attr'->Find('stripes')->Insert(@#attr);
	/Define_Tag;
	Define_Tag('Axis', -Required='Axis');
		#axis == 'y' ? Local: 'pos' = 1;
		#axis == 'y2' ? Local: 'pos'  = 2;
		#axis == 'x' ? Local: 'pos'  = 3;
		#axis == 'x2' ? Local: 'pos'  = 4;
		Params->RemoveAll('axis') & RemoveAll(-axis) & RemoveAll(#axis);
		Self->'attr' !>> 'axis' ? Self->'attr'->Insert('axis'=(Array: Map, Map, Map, Map));
		While: Self->'attr'->Find('axis')->Size < #pos;
			Self->'attr'->Find('axis')->Insert(Map);
		/While;
		Local: 'attr' = Map;
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->Find('axis')->Get(#pos)->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('AxisLabel', -Required='axis');
		Params->RemoveAll('axis') & RemoveAll(-axis) & RemoveAll(#axis);
		Local: 'attr' = Map;
		Local('_names' = (Array: 'angle'='labelangle'));
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Local: 'temp' = Map;
		#attr >> 'decimals' ? #temp->Insert('decimals'=#attr->Find('decimals'));
		#attr >> 'decimals' ? #attr->Remove('decimals');
		#attr >> 'format' ? #temp->Insert('format'=#attr->Find('format'));
		#attr >> 'format' ? #attr->Remove('format');
		#temp->size > 0 ? #attr->Insert('labelsformat'=@#temp);
		local: 'params' = Array(-Axis=#axis);
		#params->InsertFrom(#attr->Iterator);
		Self->\Axis->Run(-Owner=Self, -Params=#params);
	/Define_Tag;
	Define_Tag('AxisTitle', -Required='axis', -Required='text');
		Params->RemoveAll('axis') & RemoveAll(-axis) & RemoveAll(#axis);
		Local: 'attr' = Map;
		Local: '_names' = Array('text');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		#attr->size > 0 ? Self->Axis(-Axis=#axis, 'title'=@#attr);
	/Define_Tag;
	Define_Tag('AxisLine', -Required='axis');
		Params->RemoveAll('axis') & RemoveAll(-axis) & RemoveAll(#axis);
		Local: 'attr' = Map;
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		#attr->size > 0 ? Self->Axis(-Axis=#axis, 'line'=@#attr);
	/Define_Tag;
	Define_Tag('AxisGrid', -Required='axis', -Optional='step');
		Params->RemoveAll('axis') & RemoveAll(-axis) & RemoveAll(#axis);
		Params->RemoveAll('step') & RemoveAll(-step) & RemoveAll((local_defined: 'step') ? (local: 'step'));
		Local: 'attr' = Map;
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Local: 'params' = (Array: -Axis=#axis, -GridLines=true);
		#attr->size > 0 ? #params->Insert(-Grid=@#attr);
		Local_Defined('step') ? #params->Insert(-Step=#step);
		Self->\Axis->Run(-Owner=Self, -Params=#params);
	/Define_Tag;
	Define_Tag('AxisMinorGrid', -Required='axis', -Optional='step');
		Params->RemoveAll('axis') & RemoveAll(-axis) & RemoveAll(#axis);
		Params->RemoveAll('step') & RemoveAll(-step) & RemoveAll((local_defined: 'step') ? (local: 'step'));
		Local: 'attr' = Map;
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Local: 'params' = (Array: -Axis=#axis, -MinorGridLines=true);
		#attr->size > 0 ? #params->Insert(-MinorGrid=@#attr);
		Local_Defined('step') ? #params->Insert(-MinorStep=#step);
		Self->\Axis->Run(-Owner=Self, -Params=#params);
	/Define_Tag;
	Define_Tag('AxisStripe', -Required='axis', -Required='color', -Optional='step');
		If(Local_Defined('step'));
			Self->Axis(-Axis=#axis, -AlternateColor=#color, -Interlaced=True, -Step=#step);
		Else;
			Self->Axis(-Axis=#axis, -AlternateColor=#color, -Interlaced=True);
		/If;
	/Define_Tag;
	Define_Tag('AxisTick', -Required='axis', -Optional='tick', -Optional='step');
		If(Local_Defined('step'));
			Self->Axis(-Axis=#axis, -TickMark=(If_Null: (Local: 'tick'), 'Outer'), -Step=#step);
		Else;
			Self->Axis(-Axis=#axis, -TickMark=(If_Null: (Local: 'tick'), 'Outer'));
		/If;
	/Define_Tag;
	Define_Tag('AxisMinorTick', -Required='axis', -Optional='tick', -Optional='step');
		If(Local_Defined('step'));
			Self->Axis(-Axis=#axis, -MinorTickMark=(If_Null: (Local: 'tick'), 'Outer'), -MinorStep=#step);
		Else;
			Self->Axis(-Axis=#axis, -MinorTickMark=(If_Null: (Local: 'tick'), 'Outer'));
		/If;
	/Define_Tag;
	Define_Tag('Legend');
		Local: 'attr' = Map;
		Local: '_names' = Array('top', 'bottom', 'left', 'right', 'color'='backcolor', 'border'='toolborder', 'internal', 'external');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		#attr >> 'top' ? #attr->Insert('docked' = 'top') & Remove('top');
		#attr >> 'bottom' ? #attr->Insert('docked' = 'bottom') & Remove('bottom');
		#attr >> 'left' ? #attr->Insert('docked' = 'left') & Remove('left');
		#attr >> 'right' ? #attr->Insert('docked' = 'right') & Remove('right');
		#attr >> 'internal' ? #attr->Insert('border' = 'internal') & Remove('internal');
		#attr >> 'external' ? #attr->Insert('border' = 'external') & Remove('external');
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->Insert('legendboxobj'=@#attr);
		Self->'attr'->Insert('legendbox'=true);
	/Define_Tag;
	Define_Tag('SeriesLegend');
		Local: 'attr' = Map;
		Local: '_names' = Array('top', 'bottom', 'left', 'right', 'color'='backcolor', 'border'='toolborder', 'internal', 'external');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		#attr >> 'top' ? #attr->Insert('docked' = 'top') & Remove('top');
		#attr >> 'bottom' ? #attr->Insert('docked' = 'bottom') & Remove('bottom');
		#attr >> 'left' ? #attr->Insert('docked' = 'left') & Remove('left');
		#attr >> 'right' ? #attr->Insert('docked' = 'right') & Remove('right');
		#attr >> 'internal' ? #attr->Insert('border' = 'internal') & Remove('internal');
		#attr >> 'external' ? #attr->Insert('border' = 'external') & Remove('external');
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->Insert('serlegboxobj'=@#attr);
		Self->'attr'->Insert('serlegbox'=true);
	/Define_Tag;
	Define_Tag('series', -Required='Series');
		Local: 'pos' = Integer(#series);
		Params->RemoveAll('series') & RemoveAll(-series) & RemoveAll(#series);
		Self->'attr' !>> 'series' ? Self->'attr'->Insert('series'=(Array: Map));
		While: Self->'attr'->Find('series')->Size < #pos;
			Self->'attr'->Find('series')->Insert(Map);
		/While;
		Local: 'attr' = Map;
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Self->'attr'->Find('series')->Get(#pos)->InsertFrom(#attr->Iterator);
	/Define_Tag;
	Define_Tag('seriesmarker', -Required='Series', -Optional='shape');
		Params->RemoveAll('series') & RemoveAll(-series) & RemoveAll(#series);
		Local('attr' = Map);
		Local('_names' = Array('shape'='markershape','size'='markersize','step'='markerstep'));
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Local: 'params' = (Array: -Series=#series);
		#params->InsertFrom(#attr->Iterator);
		Self->\Series->Run(-Owner=Self, -Params=#params);
	/Define_Tag;
	Define_Tag('serieslabel', -Required='Series');
		Params->RemoveAll('series') & RemoveAll(-series) & RemoveAll(#series);
		Local: 'attr' = Map;
		Local: '_names' = Array('angle'='labelangle', 'color'='pointlabelcolor');
		Self->\GetNamed->Eval(-Owner=Self, -Params=Params);
		Self->\GetFont->Eval(-Owner=Self, -Params=Params);
		#attr >> 'font' ? #attr->Insert('pointlabelfont' = #attr->Find('font')) & Remove('font');
		Self->\GetParams->Eval(-Owner=Self, -Params=Params);
		Local: 'params' = (Array: -Series=#series);
		#params->InsertFrom(#attr->Iterator);
		Self->\Series->Run(-Owner=Self, -Params=#params);		
	/Define_Tag;
/Define_Type;

]
